<?php

namespace UltimatePostKit\Modules\HanselSlider\Widgets;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Background;
use Elementor\Utils;

use UltimatePostKit\Traits\Global_Widget_Controls;
use UltimatePostKit\Includes\Controls\GroupQuery\Group_Control_Query;
use UltimatePostKit\Modules\QueryControl\Controls\Group_Control_Posts;
use WP_Query;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

class Hansel_Slider extends Group_Control_Query {

	use Global_Widget_Controls;
		
	private $_query = null;

	public function get_name() {
		return 'upk-hansel-slider';
	}

	public function get_title() {
		return BDTUPK . esc_html__('Hansel Slider', 'ultimate-post-kit');
	}

	public function get_icon() {
		return 'upk-widget-icon upk-icon-hansel-slider';
	}

	public function get_categories() {
		return ['ultimate-post-kit-pro'];
	}

	public function get_keywords() {
		return ['post', 'carousel', 'blog', 'recent', 'news', 'slider', 'hansel'];
	}

	public function get_style_depends() {
		if ($this->upk_is_edit_mode()) {
			return ['upk-all-styles'];
		} else {
			return ['ultimate-post-kit-font', 'upk-hansel-slider'];
		}
	}

	public function get_script_depends() {
		if ( $this->upk_is_edit_mode() ) {
			return [ 'upk-all-scripts' ];
		} else {
			return [ 'upk-hansel-slider' ];
		}
	}

	public function get_custom_help_url() {
		return 'https://youtu.be/tC7WGeMQkSQ';
	}

	public function on_import($element) {
		if (!get_post_type_object($element['settings']['posts_post_type'])) {
			$element['settings']['posts_post_type'] = 'post';
		}

		return $element;
	}

	public function on_export($element) {
		$element = Group_Control_Posts::on_export_remove_setting_from_element($element, 'posts');
		return $element;
	}

	public function get_query() {
		return $this->_query;
	}

	protected function register_controls() {
		$this->start_controls_section(
			'section_content_layout',
			[
				'label' => esc_html__('Layout', 'ultimate-post-kit'),
			]
		);

		$this->add_responsive_control(
			'item_height',
			[
				'label' => esc_html__('Height', 'ultimate-post-kit'),
				'type'  => Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'vh' ],
				'range' => [
					'px' => [
						'min' => 200,
						'max' => 1080,
					],
					'vh' => [
						'min' => 10,
						'max' => 100,
					],
				],
				'selectors'   => [
					'{{WRAPPER}} .upk-hansel-slider .upk-item' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'content_width',
			[
				'label' => esc_html__('Content Max Width', 'ultimate-post-kit'),
				'type'  => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 200,
						'max' => 1200,
					],
				],
				'selectors'   => [
					'{{WRAPPER}} .upk-hansel-slider .upk-content' => 'max-width: {{SIZE}}px;',
				],
			]
		);

		$this->add_control(
			'content_position',
			[
				'label'   => __( 'Content Position', 'ultimate-post-kit' ),
				'type'    => Controls_Manager::CHOOSE,
				'default' => 'center',
				'options' => [
					'top'   => [
						'title' => __( 'Top', 'ultimate-post-kit' ),
						'icon'  => 'eicon-v-align-top',
					],
					'center' => [
						'title' => __( 'Center', 'ultimate-post-kit' ),
						'icon'  => 'eicon-v-align-middle',
					],
					'bottom'  => [
						'title' => __( 'Bottom', 'ultimate-post-kit' ),
						'icon'  => 'eicon-v-align-bottom',
					],
				],
				'prefix_class' => 'upk-content-position--',
				'render_type' => 'template',
				'toggle' => false
			]
		);

		$this->add_control(
			'content_alignment',
			[
				'label'   => __( 'Alignment', 'ultimate-post-kit' ),
				'type'    => Controls_Manager::CHOOSE,
				'default' => 'left',
				'options' => [
					'left'   => [
						'title' => __( 'Left', 'ultimate-post-kit' ),
						'icon'  => 'eicon-h-align-left',
					],
					'right'  => [
						'title' => __( 'Right', 'ultimate-post-kit' ),
						'icon'  => 'eicon-h-align-right',
					],
				],
				'prefix_class' => 'upk-content-align--',
				'render_type' => 'template',
				'toggle' => false,
			]
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
				'name'      => 'primary_thumbnail',
				'exclude'   => ['custom'],
				'default'   => 'full',
			]
		);

		$this->add_control(
			'hr_1',
			[
				'type'    => Controls_Manager::DIVIDER,
			]
		);

		//Global Title Controls
		$this->register_title_controls();

		$this->add_control(
			'show_category',
			[
				'label'   => esc_html__('Show Category', 'ultimate-post-kit'),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);

		$this->add_control(
			'show_excerpt',
			[
				'label'   => esc_html__( 'Show Text', 'ultimate-post-kit' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);
		
		$this->add_control(
			'excerpt_length',
			[
				'label'       => esc_html__( 'Text Limit', 'ultimate-post-kit' ),
				'description' => esc_html__( 'It\'s just work for main content, but not working with excerpt. If you set 0 so you will get full main content.', 'ultimate-post-kit' ),
				'type'        => Controls_Manager::NUMBER,
				'default'     => 15,
				'condition'   => [
					'show_excerpt' => 'yes'
				],
			]
		);
		
		$this->add_control(
			'strip_shortcode',
			[
				'label'     => esc_html__( 'Strip Shortcode', 'ultimate-post-kit' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => [
					'show_excerpt' => 'yes',
				],
			]
		);

		$this->add_control(
			'show_author',
			[
				'label'     => esc_html__( 'Show Author', 'ultimate-post-kit' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'separator' => 'before'
			]
		);
		
		$this->add_control(
			'show_comments',
			[
				'label' => esc_html__( 'Show Comments', 'ultimate-post-kit' ),
				'type'  => Controls_Manager::SWITCHER,
			]
		);
		
		$this->add_control(
			'meta_separator',
			[
				'label'       => __( 'Separator', 'ultimate-post-kit' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => '//',
				'label_block' => false,
			]
		);
		
		//Global Date Controls
		$this->register_date_controls();

		$this->end_controls_section();

		// Query Settings
		$this->start_controls_section(
			'section_post_query_builder',
			[
				'label' => __( 'Query', 'ultimate-post-kit' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'item_limit',
			[
				'label' => esc_html__('Item Limit', 'ultimate-post-kit'),
				'type'  => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 1,
						'max' => 20,
					],
				],
				'default' => [
					'size' => 3,
				],
			]
		);
		
		$this->register_query_builder_controls();
		
		$this->end_controls_section();

		$this->start_controls_section(
			'section_carousel_settings',
			[
				'label' => __( 'Slider Settings', 'ultimate-post-kit' ),
			]
		);

		$this->add_control(
			'autoplay',
			[
				'label'   => __( 'Autoplay', 'ultimate-post-kit' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
				
			]
		);

		$this->add_control(
			'autoplay_speed',
			[
				'label'     => esc_html__( 'Autoplay Speed', 'ultimate-post-kit' ),
				'type'      => Controls_Manager::NUMBER,
				'default'   => 5000,
				'condition' => [
					'autoplay' => 'yes',
				],
			]
		);

		$this->add_control(
			'pauseonhover',
			[
				'label' => esc_html__( 'Pause on Hover', 'ultimate-post-kit' ),
				'type'  => Controls_Manager::SWITCHER,
			]
		);

		$this->add_control(
			'grab_cursor',
			[
				'label'   => __( 'Grab Cursor', 'ultimate-post-kit' ),
				'type'    => Controls_Manager::SWITCHER,
			]
		);

		$this->add_control(
			'loop',
			[
				'label'   => __( 'Loop', 'ultimate-post-kit' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
				
			]
		);


		$this->add_control(
			'speed',
			[
				'label'   => __( 'Animation Speed (ms)', 'ultimate-post-kit' ),
				'type'    => Controls_Manager::SLIDER,
				'default' => [
					'size' => 500,
				],
				'range' => [
					'px' => [
						'min'  => 100,
						'max'  => 5000,
						'step' => 50,
					],
				],
			]
		);

		$this->add_control(
			'observer',
			[
				'label'       => __( 'Observer', 'ultimate-post-kit' ),
				'description' => __( 'When you use carousel in any hidden place (in tabs, accordion etc) keep it yes.', 'ultimate-post-kit' ),
				'type'        => Controls_Manager::SWITCHER,				
			]
		);

		$this->end_controls_section();

		//Style
		$this->start_controls_section(
			'upk_section_style',
			[
				'label' => esc_html__('Items', 'ultimate-post-kit'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'item_background',
				'selector'  => '{{WRAPPER}} .upk-hansel-slider .upk-item',
			]
		);

		$this->add_responsive_control(
			'item_padding',
			[
				'label' 	 => __('Padding', 'ultimate-post-kit'),
				'type' 		 => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-hansel-slider .upk-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_title',
			[
				'label'     => esc_html__('Title', 'ultimate-post-kit'),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_title' => 'yes',
				],
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-hansel-slider .upk-title a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'title_hover_color',
			[
				'label'     => esc_html__('Hover Color', 'ultimate-post-kit'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-hansel-slider .upk-title a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'title_background',
				'selector'  => '{{WRAPPER}} .upk-hansel-slider .upk-title a',
			]
		);

		$this->add_responsive_control(
			'title_spacing',
			[
				'label'      => esc_html__('Spacing', 'ultimate-post-kit'),
				'type'       => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min'  => 0,
						'max'  => 100,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .upk-hansel-slider .upk-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'      => 'title_typography',
				'label'     => esc_html__('Typography', 'ultimate-post-kit'),
				'selector'  => '{{WRAPPER}} .upk-hansel-slider .upk-title',
			]
		);

		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name' => 'title_text_shadow',
				'label' => __('Text Shadow', 'ultimate-post-kit'),
				'selector' => '{{WRAPPER}} .upk-hansel-slider .upk-title a',
			]
		);

		$this->end_controls_section();

		
		$this->start_controls_section(
			'section_style_text',
			[
				'label'     => esc_html__('Text', 'ultimate-post-kit'),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_excerpt' => 'yes',
				],
			]
		);

		$this->add_control(
			'text_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-hansel-slider .upk-text' => 'color: {{VALUE}};',
				],
			]
		);


		$this->add_responsive_control(
			'text_margin',
			[
				'label' 	 => __('Margin', 'ultimate-post-kit'),
				'type' 		 => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-hansel-slider .upk-text' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'text_width',
			[
				'label' => esc_html__('Max Width', 'ultimate-post-kit'),
				'type'  => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 100,
						'max' => 800,
					],
				],
				'selectors'   => [
					'{{WRAPPER}} .upk-hansel-slider .upk-text' => 'max-width: {{SIZE}}px;',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'      => 'text_typography',
				'label'     => esc_html__('Typography', 'ultimate-post-kit'),
				'selector'  => '{{WRAPPER}} .upk-hansel-slider .upk-text',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_meta',
			[
				'label'      => esc_html__( 'Meta', 'ultimate-post-kit' ),
				'tab'        => Controls_Manager::TAB_STYLE,
				'conditions' => [
					'relation' => 'or',
					'terms'    => [
						[
							'name'  => 'show_author',
							'value' => 'yes'
						],
						[
							'name'  => 'show_date',
							'value' => 'yes'
						],
						[
							'name'  => 'show_comments',
							'value' => 'yes'
						]
					]
				],
			]
		);
		
		$this->add_control(
			'meta_color',
			[
				'label'     => esc_html__( 'Color', 'ultimate-post-kit' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-hansel-slider .upk-meta, {{WRAPPER}} .upk-hansel-slider .upk-meta .upk-author-wrap .upk-name' => 'color: {{VALUE}};',
				],
			]
		);
		
		$this->add_control(
			'meta_hover_color',
			[
				'label'     => esc_html__( 'Hover Color', 'ultimate-post-kit' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-hansel-slider .upk-meta .upk-author-wrap .upk-name:hover' => 'color: {{VALUE}};',
				],
			]
		);
		
		$this->add_responsive_control(
			'meta_spacing',
			[
				'label'     => esc_html__( 'Spacing', 'ultimate-post-kit' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-hansel-slider .upk-meta' => 'padding-top: {{SIZE}}{{UNIT}};',
				],
			]
		);
		
		$this->add_responsive_control(
			'meta_space_between',
			[
				'label'     => esc_html__( 'Space Between', 'ultimate-post-kit' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-hansel-slider .upk-meta .upk-separator' => 'margin: 0 {{SIZE}}{{UNIT}};',
				],
			]
		);
		
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'meta_typography',
				'label'    => esc_html__( 'Typography', 'ultimate-post-kit' ),
				'selector' => '{{WRAPPER}} .upk-hansel-slider .upk-meta',
			]
		);
		
		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_category',
			[
				'label'     => esc_html__('Category', 'ultimate-post-kit'),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_category' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'category_bottom_spacing',
			[
				'label'   => esc_html__('Spacing', 'ultimate-post-kit'),
				'type'    => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min'  => 0,
						'max'  => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-hansel-slider .upk-category' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs('tabs_category_style');

		$this->start_controls_tab(
			'tab_category_normal',
			[
				'label' => esc_html__('Normal', 'ultimate-post-kit'),
			]
		);

		$this->add_control(
			'category_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-hansel-slider .upk-category a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'category_background',
				'selector'  => '{{WRAPPER}} .upk-hansel-slider .upk-category a',
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'        => 'category_border',
				'selector'    => '{{WRAPPER}} .upk-hansel-slider .upk-category a',
			]
		);

		$this->add_responsive_control(
			'category_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-hansel-slider .upk-category a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'category_padding',
			[
				'label'      => esc_html__('Padding', 'ultimate-post-kit'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-hansel-slider .upk-category a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'category_spacing',
			[
				'label'   => esc_html__('Space Between', 'ultimate-post-kit'),
				'type'    => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min'  => 0,
						'max'  => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-hansel-slider .upk-category a+a' => 'margin-left: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'category_shadow',
				'selector' => '{{WRAPPER}} .upk-hansel-slider .upk-category a',
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'category_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit'),
				'selector' => '{{WRAPPER}} .upk-hansel-slider .upk-category a',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_category_hover',
			[
				'label' => esc_html__('Hover', 'ultimate-post-kit'),
				'condition' => [
					'show_category' => 'yes'
				]
			]
		);

		$this->add_control(
			'category_hover_color',
			[
				'label'     => esc_html__('Color', 'ultimate-post-kit'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-hansel-slider .upk-category a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'category_hover_background',
				'selector'  => '{{WRAPPER}} .upk-hansel-slider .upk-category a:hover',
			]
		);

		$this->add_control(
			'category_hover_border_color',
			[
				'label'     => esc_html__('Border Color', 'ultimate-post-kit'),
				'type'      => Controls_Manager::COLOR,
				'condition' => [
					'category_border_border!' => '',
				],
				'selectors' => [
					'{{WRAPPER}} .upk-hansel-slider .upk-category a:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		//Navigation Css
		$this->start_controls_section(
			'section_style_navigation',
			[
				'label'     => __( 'Navigation', 'ultimate-post-kit' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'navigation_background',
				'selector'  => '{{WRAPPER}} .upk-hansel-pagination',
			]
		);

		$this->add_control(
			'arrows_heading',
			[
				'label'     => __( 'A R R O W S', 'ultimate-post-kit' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_control(
			'nav_arrows_icon',
			[
				'label'   => esc_html__( 'Arrows Icon', 'ultimate-post-kit' ),
				'type'    => Controls_Manager::SELECT,
				'default' => '0',
				'options' => [
					'0' => esc_html__( 'Default', 'bdthemes-element-pack' ),
					'1' => esc_html__('Style 1', 'ultimate-post-kit'),
					'2' => esc_html__('Style 2', 'ultimate-post-kit'),
					'3' => esc_html__('Style 3', 'ultimate-post-kit'),
					'4' => esc_html__('Style 4', 'ultimate-post-kit'),
					'5' => esc_html__('Style 5', 'ultimate-post-kit'),
					'6' => esc_html__('Style 6', 'ultimate-post-kit'),
					'7' => esc_html__('Style 7', 'ultimate-post-kit'),
					'8' => esc_html__('Style 8', 'ultimate-post-kit'),
					'9' => esc_html__('Style 9', 'ultimate-post-kit'),
					'10' => esc_html__('Style 10', 'ultimate-post-kit'),
					'11' => esc_html__('Style 11', 'ultimate-post-kit'),
					'12' => esc_html__('Style 12', 'ultimate-post-kit'),
					'13' => esc_html__('Style 13', 'ultimate-post-kit'),
					'14' => esc_html__('Style 14', 'ultimate-post-kit'),
					'15' => esc_html__('Style 15', 'ultimate-post-kit'),
					'16' => esc_html__('Style 16', 'ultimate-post-kit'),
					'17' => esc_html__('Style 17', 'ultimate-post-kit'),
					'18' => esc_html__('Style 18', 'ultimate-post-kit'),
					'circle-1' => esc_html__('Style 19', 'ultimate-post-kit'),
					'circle-2' => esc_html__('Style 20', 'ultimate-post-kit'),
					'circle-3' => esc_html__('Style 21', 'ultimate-post-kit'),
					'circle-4' => esc_html__('Style 22', 'ultimate-post-kit'),
					'square-1' => esc_html__('Style 23', 'ultimate-post-kit'),
				],
			]
		);

		$this->add_control(
			'arrows_color',
			[
				'label'     => __( 'Color', 'ultimate-post-kit' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-hansel-pagination .upk-navigation-next, 
					{{WRAPPER}}  .upk-hansel-pagination .upk-navigation-prev' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'arrows_typography',
				'label'    => esc_html__('Typography', 'ultimate-post-kit'),
				'selector' => '{{WRAPPER}} .upk-hansel-pagination .upk-navigation-next, {{WRAPPER}}  
				.upk-hansel-pagination .upk-navigation-prev',
			]
		);

		$this->add_control(
			'dots_heading',
			[
				'label'     => __( 'D O T S', 'ultimate-post-kit' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_control(
			'dots_color',
			[
				'label'     => __( 'Color', 'ultimate-post-kit' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-hansel-pagination .swiper-pagination .swiper-pagination-bullet' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'active_dot_color',
			[
				'label'     => __( 'Active Color', 'ultimate-post-kit' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .upk-hansel-pagination .swiper-pagination .swiper-pagination-bullet-active' => 'background-color: {{VALUE}}',
				],
			]
		);

		
		$this->add_responsive_control(
			'dots_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'ultimate-post-kit'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .upk-hansel-pagination .swiper-pagination .swiper-pagination-bullet' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'dots_height_size',
			[
				'label' => __( 'Height(px)', 'ultimate-post-kit' ),
				'type'  => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 5,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-hansel-pagination .swiper-pagination .swiper-pagination-bullet' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'dots_width_size',
			[
				'label' => __( 'Width(px)', 'ultimate-post-kit' ),
				'type'  => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 5,
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .upk-hansel-pagination .swiper-pagination .swiper-pagination-bullet' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Main query render for this widget
	 * @param $posts_per_page number item query limit
	 */
	public function query_posts( $posts_per_page ) {
		
		$default = $this->getGroupControlQueryArgs();
		if ( $posts_per_page ) {
			$args['posts_per_page'] = $posts_per_page;
				$args['paged']  = max( 1, get_query_var( 'paged' ), get_query_var( 'page' ) );
		}
		$args         = array_merge( $default, $args );
		$this->_query = new WP_Query( $args );
	}

	public function render_image($image_id, $size) {
		$placeholder_image_src = Utils::get_placeholder_image_src();

		$image_src = wp_get_attachment_image_src($image_id, $size);

		if (!$image_src) {
			$image_src = $placeholder_image_src;
		} else {
			$image_src = $image_src[0];
		}

		?>

		<img class="upk-img swiper-lazy" src="<?php echo esc_url($image_src); ?>" alt="<?php echo esc_attr(get_the_title()); ?>">

		<?php
	}

	public function render_title() {
		$settings = $this->get_settings_for_display();

		if (!$this->get_settings('show_title')) {
			return;
		}

		$this->add_render_attribute('slider-title', 'class', 'upk-title', true);
		$titleClass = $this->get_render_attribute_string('slider-title');
		echo
		'<' . esc_html($settings['title_tags']) . ' ' . $titleClass . ' >
				<a href="' . esc_url(get_permalink()) . '" title="' . esc_attr(get_the_title()) . '">
					' . esc_html(get_the_title())  . '
				</a>
			</' . esc_html($settings['title_tags']) . '>';

	}

	public function render_excerpt( $excerpt_length ) {
			
		if ( ! $this->get_settings( 'show_excerpt' ) ) {
			return;
		}
		$strip_shortcode = $this->get_settings_for_display( 'strip_shortcode' );
		?>
		<div class="upk-text" data-swiper-parallax-x="-100">
			<?php
				if ( has_excerpt() ) {
					the_excerpt();
				} else {
					echo ultimate_post_kit_custom_excerpt( $excerpt_length, $strip_shortcode );
				}
			?>
		</div>
		
		<?php
	}

	public function render_category() {

		if (!$this->get_settings('show_category')) {
			return;
		}
		?>
		<div class="upk-category" data-swiper-parallax-x="-300">
			<?php echo upk_get_category($this->get_settings('posts_source')); ?>
		</div>
		<?php
	}

	public function render_date() {
		$settings = $this->get_settings_for_display();
		
		
		if ( ! $this->get_settings( 'show_date' ) ) {
			return;
		}
		
		?>
		<div class="upk-flex upk-flex-middle">
			<div class="upk-scott-list-date">
				<?php if ( $settings['human_diff_time'] == 'yes' ) {
					echo ultimate_post_kit_post_time_diff( ( $settings['human_diff_time_short'] == 'yes' ) ? 'short' : '' );
				} else {
					echo get_the_date();
				} ?>
			</div>
			<?php if ($settings['show_time']) : ?>
			<div class="upk-post-time">
				<i class="upk-icon-clock" aria-hidden="true"></i>
				<?php echo get_the_time(); ?>
			</div>
			<?php endif; ?>
		</div>
		
		<?php
	}

	public function render_author() {
			
		if ( ! $this->get_settings( 'show_author' ) ) {
			return;
		}
		?>
		<div class="upk-author-wrap">
			<span class="upk-by"><?php echo esc_html_x('by', 'Frontend', 'ultimate-post-kit') ?></span>
			<a class="upk-name" href="<?php echo get_author_posts_url( get_the_author_meta( 'ID' ) ) ?>">
				<?php echo get_the_author() ?>
			</a>
		</div>
		<?php
	}
	
	public function render_comments( $id = 0 ) {
		
		if ( ! $this->get_settings( 'show_comments' ) ) {
			return;
		}
		?>

		<div class="upk-scott-comments">
			<?php echo get_comments_number( $id ) ?>
			<?php echo esc_html__( 'Comments', 'ultimate-post-kit' ) ?>
		</div>
		
		<?php
	}

	public function render_header() {
		$id              = 'upk-hansel-slider-' . $this->get_id();
		$settings        = $this->get_settings_for_display();
		
		$this->add_render_attribute( 'hansel-slider', 'id', $id );
		$this->add_render_attribute( 'hansel-slider', 'class', ['upk-hansel-slider'] );

		$this->add_render_attribute(
			[
				'hansel-slider' => [
					'data-settings' => [
						wp_json_encode(array_filter([
							"autoplay"       => ( "yes" == $settings["autoplay"] ) ? [ "delay" => $settings["autoplay_speed"] ] : false,
							"loop"           => ($settings["loop"] == "yes") ? true : false,
							"speed"          => $settings["speed"]["size"],
							"effect"         => 'fade',
							"lazy"           => true,
							"parallax"       => true,
							"grabCursor"     => ($settings["grab_cursor"] === "yes") ? true : false,
							"pauseOnHover"   => ("yes" == $settings["pauseonhover"]) ? true : false,
							"slidesPerView"  => 1,
							"observer"       => ($settings["observer"]) ? true : false,
							"observeParents" => ($settings["observer"]) ? true : false,
			      	        "navigation" => [
			      				"nextEl" => "#" . $id . " .upk-navigation-next",
			      				"prevEl" => "#" . $id . " .upk-navigation-prev",
			      			],
			      			"pagination" => [
								"el"             => "#" . $id . " .swiper-pagination",
								"clickable"      => "true",
							],
							"lazy" => [
								"loadPrevNext"  => "true",
							],
				        ]))
					]
				]
			]
		);

		?>
		<div <?php echo $this->get_render_attribute_string( 'hansel-slider' ); ?>>
			<div class="upk-hansel-slider-wrapper">
				<div class="swiper-container">
					<div class="swiper-wrapper">
		<?php
	}

	public function render_footer() {
		$settings = $this->get_settings_for_display();
		
		?>
					</div>
					<div class="upk-hansel-pagination">

						<a href="" class="upk-navigation-prev">
							<i class="upk-icon-arrow-left-<?php echo esc_attr($settings['nav_arrows_icon']); ?>" aria-hidden="true"></i>
							<span>prev</span>
						</a>
						<a href="" class="upk-navigation-next">
							<i class="upk-icon-arrow-right-<?php echo esc_attr($settings['nav_arrows_icon']); ?>" aria-hidden="true"></i>
							<span>next</span>
						</a>

						<div class="swiper-pagination"></div>

					</div>
				</div>
				
			</div>
		</div>

		<?php
	}

	public function render_post_grid_item($post_id, $image_size, $excerpt_length) {
		$settings = $this->get_settings_for_display();

		$this->add_render_attribute('slider-item', 'class', 'upk-item swiper-slide upk-transition-toggle', true);

		?>

		<div <?php echo $this->get_render_attribute_string('slider-item'); ?>>
			<div class="upk-item-box">
				<div class="upk-img-wrap">
					<?php $this->render_image(get_post_thumbnail_id($post_id), $image_size); ?>
				</div>
				<div class="upk-content">

					<?php $this->render_category(); ?>

					<div data-swiper-parallax-x="-200">
						<?php $this->render_title(); ?>
					</div>
					
					<?php $this->render_excerpt( $excerpt_length ); ?>

					<?php if ( $settings['show_author'] or $settings['show_comments'] or $settings['show_date'] ) : ?>
						<div class="upk-meta upk-flex upk-flex-middle" data-swiper-parallax-x="-80">
							<?php if ( $settings['show_author'] ) : ?>
								<?php $this->render_author(); ?>
								<span class="upk-separator"><?php echo $settings['meta_separator']; ?></span>
							<?php endif; ?>
							<?php $this->render_date(); ?>
							<?php if ( $settings['show_comments'] ) : ?>
								<span class="upk-separator"><?php echo $settings['meta_separator']; ?></span>
								<?php $this->render_comments( $post_id ); ?>
							<?php endif; ?>
						</div>
					<?php endif; ?>

				</div>
			</div>
		</div>
		<?php
	}

	public function render() {
		$settings = $this->get_settings_for_display();

		$this->query_posts($settings['item_limit']['size']);
		$wp_query = $this->get_query();

		if (!$wp_query->found_posts) {
			return;
		}
		
		$this->render_header();

		while ( $wp_query->have_posts() ) {
			$wp_query->the_post();
			$thumbnail_size = $settings['primary_thumbnail_size'];

			$this->render_post_grid_item( get_the_ID(), $thumbnail_size, $settings['excerpt_length'] );
		}

		$this->render_footer();

		wp_reset_postdata();
	}
}
